<?php
// ultra_fast_init.php
// Versão: Ultra-rápida com cache APCu (quando disponível) + fallback em memória
// Mantém 100% da lógica original, minimiza acesso ao DB na inicialização usando cache

ini_set("display_errors", 1);
include __DIR__ . "/../includes/functions.php";

// ------------------------
// Configurações de cache
// ------------------------
$APCU_ENABLED = function_exists('apcu_fetch') && ini_get('apc.enabled');
$CACHE_TTL = 300; // segundos (5 minutos) - ajustar conforme necessário

function cache_fetch($key) {
    global $APCU_ENABLED;
    static $local = [];
    if (isset($local[$key])) return $local[$key];
    if ($APCU_ENABLED) {
        $val = apcu_fetch($key);
        $local[$key] = $val;
        return $val;
    }
    return null;
}

function cache_store($key, $value, $ttl = 300) {
    global $APCU_ENABLED;
    static $local = [];
    $local[$key] = $value;
    if ($APCU_ENABLED) {
        apcu_store($key, $value, $ttl);
    }
}

// ------------------------
// Ler JSON de entrada
// ------------------------
$jsonIn = file_get_contents("php://input");
$response = json_decode($jsonIn, true);
$decoded = base64_decode($response["data"]);
$authData = json_decode($decoded, true);
$portal = [];

// ------------------------
// Verificar MAC
// ------------------------
if (isset($authData["app_device_id"])) {
    $macAddress = base64_decode($authData["app_device_id"]);
    $macAddress = substr($macAddress, 0, 12);
    $formattedMac = strtoupper(preg_replace("/..(?!$)/", "\\$0:", $macAddress));
} else {
    $formattedMac = strtoupper($authData["mac_address"]);
}

// ------------------------
// Carregamento agressivo de cache: dns, themes, settings, playlists e language.json
// Tentamos buscar no APCu (persistente entre requests). Se não existir, carregamos do DB e gravamos no cache.
// ------------------------
$dns_cache_key = 'studiolive_dns_cache_v1';
$themes_cache_key = 'studiolive_themes_cache_v1';
$settings_cache_key = 'studiolive_settings_cache_v1';
$playlist_cache_key = 'studiolive_playlist_cache_v1';
$lang_cache_key = 'studiolive_language_cache_v1';

$dns_all = cache_fetch($dns_cache_key);
$themes_all = cache_fetch($themes_cache_key);
$settings_all = cache_fetch($settings_cache_key);
$playlist_all = cache_fetch($playlist_cache_key);
$lang_all = cache_fetch($lang_cache_key);

// Função utilitária para forçar recarga do cache (pode ser chamada manualmente se necessário)
function refresh_caches() {
    global $db, $dns_cache_key, $themes_cache_key, $settings_cache_key, $playlist_cache_key, $lang_cache_key, $CACHE_TTL;
    $dns_all_f = $db->select('dns', '*');
    cache_store($dns_cache_key, $dns_all_f, $CACHE_TTL);

    $themes_all_f = $db->select('themes', '*');
    cache_store($themes_cache_key, $themes_all_f, $CACHE_TTL);

    $settings_all_f = $db->select('settings', '*');
    cache_store($settings_cache_key, $settings_all_f, $CACHE_TTL);

    $playlist_all_f = $db->select('playlist', '*');
    cache_store($playlist_cache_key, $playlist_all_f, $CACHE_TTL);

    $lang_all_f = json_decode(file_get_contents('language.json'), true);
    cache_store($lang_cache_key, $lang_all_f, $CACHE_TTL);

    return true;
}

// Se algum cache estiver vazio, recarregamos apenas o que está faltando
$need_refresh = false;
if ($dns_all === null || $themes_all === null || $settings_all === null || $playlist_all === null || $lang_all === null) {
    $need_refresh = true;
}

if ($need_refresh) {
    // Carrega tudo do banco (apenas quando necessário)
    $dns_all = $db->select('dns', '*');
    cache_store($dns_cache_key, $dns_all, $CACHE_TTL);

    $themes_all = $db->select('themes', '*');
    cache_store($themes_cache_key, $themes_all, $CACHE_TTL);

    $settings_all = $db->select('settings', '*', "id = :id", '', [':id' => 1]);
    cache_store($settings_cache_key, $settings_all, $CACHE_TTL);

    // Carrega todas as playlists em cache (atenção: pode ser grande, ajustar TTL/conteúdo conforme infra)
    $playlist_all = $db->select('playlist', '*');
    cache_store($playlist_cache_key, $playlist_all, $CACHE_TTL);

    $lang_all = json_decode(file_get_contents('language.json'), true);
    cache_store($lang_cache_key, $lang_all, $CACHE_TTL);
}

// Indexar dns por id para acesso rápido
$dns_indexed = [];
if (!empty($dns_all)) {
    foreach ($dns_all as $d) {
        $dns_indexed[$d['id']] = $d;
    }
}

// Indexar playlists por mac_address para evitar consultas ao DB
$playlists_by_mac = [];
if (!empty($playlist_all)) {
    foreach ($playlist_all as $pl) {
        $macKey = strtoupper($pl['mac_address']);
        if (!isset($playlists_by_mac[$macKey])) $playlists_by_mac[$macKey] = [];
        $playlists_by_mac[$macKey][] = $pl;
    }
}

// ------------------------
// Gerar ou recuperar device_key
// (mantemos a lógica original: consultamos DB se não houver dispositivo persistente no cache)
// ------------------------
$existingKey = $db->select("device_keys", "device_key", "mac_address = :mac_address", "", [":mac_address" => $formattedMac]);

if (!empty($existingKey)) {
    $device_key = $existingKey[0]["device_key"];
} else {
    $device_key = substr(md5($formattedMac), 0, 8);
    $db->insert('device_keys', [
        'mac_address' => $formattedMac,
        'device_key' => $device_key
    ]);
    // Opcional: atualizar cache de playlists/device info se necessário (não obrigatório aqui)
}

// ------------------------
// Status do chatbot (carregado do cache/settings se possível)
// ------------------------
$res_bot = $db->select("chatbot", "bot_status, bot_url, bot_dns", "id = :id", "", [':id' => 1]);
$bot_status = $res_bot[0]["bot_status"] ?? 0;
$bot_url = $res_bot[0]["bot_url"] ?? '';
$bot_dns_id = $res_bot[0]["bot_dns"] ?? '';

// ------------------------
// Verificar cadastro na playlist: usamos o cache de playlists para leitura inicial (sem DB)
// Se não encontrado e chatbot ativo, mantemos o comportamento original (insere no DB)
// ------------------------
$existingClient = $playlists_by_mac[$formattedMac] ?? [];

if (empty($existingClient) && $bot_status == 1) {
    // Chamamos o chatbot de forma assíncrona para não bloquear (mantemos inserir no DB quando responder)
    asyncCurlPostBackground($bot_url, json_encode([
        'receiveMessageAppId' => 'com.whatsapp',
        'receiveMessagePattern' => ['*'],
        'senderName' => 'API DE CADASTRO',
        'groupName' => '',
        'senderMesage' => 'api_cadastro',
        'senderMessage' => 'api_cadastro',
        'messageDateTime' => time(),
        'isMessageFromGroup' => false
    ]));
}

// ------------------------
// Construir portal (urls) usando apenas o cache em memória — sem acessar o DB na inicialização
// ------------------------
$results = $playlists_by_mac[$formattedMac] ?? [];
foreach ($results as $result) {
    if (isset($dns_indexed[$result['dns_id']])) {
        $row = $dns_indexed[$result['dns_id']];
        $portal[] = [
            "is_protected" => 0,
            "id" => $row["id"],
            "url" => $row["url"] . "/get.php?username=" . $result["username"] . "&password=" . $result["password"] . "&type=m3u_plus&output=ts",
            "name" => $row["title"],
            "playlist_name" => $row["title"],
            "type" => "xc",
            "created_at" => $result["created_at"] ?? date('Y-m-d H:i:s'),
            "updated_at" => $result["updated_at"] ?? date('Y-m-d H:i:s')
        ];
    }
}

// ------------------------
// Outros dados (themes/settings/languages)
// ------------------------
$backdropUrl = getBaseUrl();
$settings = $settings_all[0] ?? ['note_title' => '', 'note_content' => ''];
$theme = $themes_all[0] ?? ['theme_id' => '1'];
$theme_id = !empty($theme) ? $theme['theme_id'] : '1';
$result_pin = $results[0] ?? null;

// ------------------------
// Funções utilitárias: language cache, async background POST, getBaseUrl
// ------------------------
function getLanguages() {
    global $lang_all;
    return $lang_all ?? json_decode(file_get_contents('language.json'), true);
}

function asyncCurlPostBackground($url, $payload) {
    // Envia um POST em background sem bloquear usando fsockopen com non-blocking write
    $parts = parse_url($url);
    $scheme = $parts['scheme'] ?? 'http';
    $host = $parts['host'] ?? '';
    $port = isset($parts['port']) ? $parts['port'] : ($scheme === 'https' ? 443 : 80);
    $path = ($parts['path'] ?? '/') . (isset($parts['query']) ? '?' . $parts['query'] : '');

    $errno = 0;
    $errstr = '';
    $timeout = 0.1; // super curto: não bloqueia

    $transport = ($scheme === 'https') ? 'ssl://' : '';

    $fp = @fsockopen($transport . $host, $port, $errno, $errstr, $timeout);
    if (!$fp) return false;

    stream_set_blocking($fp, 0);

    $out = "POST {$path} HTTP/1.1\r\n";
    $out .= "Host: {$host}\r\n";
    $out .= "Content-Type: application/json\r\n";
    $out .= "Content-Length: " . strlen($payload) . "\r\n";
    $out .= "Connection: Close\r\n\r\n";
    $out .= $payload;

    fwrite($fp, $out);
    // não espera resposta
    fclose($fp);
    return true;
}

function getBaseUrl() {
    $protocol = (!empty($_SERVER["HTTPS"]) && $_SERVER["HTTPS"] !== "off" || $_SERVER["SERVER_PORT"] == 443) ? "https://" : "http://";
    $domainName = $_SERVER["HTTP_HOST"] ?? 'localhost';
    $folderPath = dirname($_SERVER["PHP_SELF"] ?? '/');
    return $protocol . $domainName . $folderPath;
}

// ------------------------
// Response final (mantendo exatamente os campos originais)
// ------------------------
$response_out = [
    "android_version_code" => "1.0.0",
    "apk_url" => "",
    "device_key" => $device_key,
    "expire_date" => "2034-03-26",
    "is_google_paid" => true,
    "is_trial" => 0,
    "languages" => getLanguages(),
    "mac_registered" => true,
    "themes" => [],
    "trial_days" => 7,
    "plan_id" => "36269518",
    "mac_address" => $formattedMac,
    "pin" => $result_pin['pin'] ?? '0000',
    "price" => "7.99",
    "apk_link" => "",
    "urls" => $portal,
    "note_title" => $settings['note_title'] ?? '',
    "note_content" => $settings['note_content'] ?? '',
    "qr_url" => "",
    "qr_url_short" => "",
    "home_mode" => null,
    "home_url1" => $backdropUrl,
    "home_url2" => $backdropUrl,
    "theme" => $theme_id
];

echo Encryption::run(json_encode($response_out), "IBO_38");
?>
